// $Id: CMenuItemDisplay.cpp,v 1.10 2007/02/08 21:07:54 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CMenuItemDisplay.hpp"
#include "CSelectableMenuItem.hpp"
#include <Exceptions/CException.hpp>
using Exponent::GUI::Controls::CMenuItemDisplay;
using Exponent::GUI::Controls::CSelectableMenuItem;
using Exponent::Exceptions::CException;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CMenuItemDisplay, CControlPanel);

//	===========================================================================
CMenuItemDisplay::CMenuItemDisplay(IControlRoot *root, 
								   const long uniqueId, 
								   const CRect &area, 
								   CMenuItem *menuItem, 
								   const bool useIcons, 
								   const bool selectable, 
								   const bool subMenu, 
								   const long iconSize) 
				: CControlPanel(root->getParentWindow(), root, uniqueId, area, NULL)
				, m_mouseOver(false)
				, m_useIcons(useIcons)
				, m_hasSubMenu(subMenu)
				, m_hasSelect(selectable)
				, m_iconSize(iconSize)
{
	EXPONENT_CLASS_CONSTRUCTION(CMenuItemDisplay);
	
	// Store the default colours
	m_colours.m_disabledTextColour   = CAlphaColour::CALPHACOLOUR_LIGHT_GREY;
	m_colours.m_frameColour		     = CAlphaColour::CALPHACOLOUR_PURPLE;
	m_colours.m_selectedColour	     = CAlphaColour::CALPHACOLOUR_PURPLE;
	m_colours.m_seperatorColour	     = CAlphaColour::CALPHACOLOUR_ORANGE;
	m_colours.m_subMenuPointerColour = CAlphaColour::CALPHACOLOUR_BLACK;
	m_colours.m_textColour			 = CAlphaColour::CALPHACOLOUR_BLACK;
	
	m_imageLabel        = new CImageLabel(			  this, CControl::CCONTROL_NO_ID_REQUIRED, CRect());
	m_titleLabel        = new CTextLabel(			  this, CControl::CCONTROL_NO_ID_REQUIRED, CRect(), " ");
	m_subMenuPointer    = new CMenuItemSubMenuPointer(this, CControl::CCONTROL_NO_ID_REQUIRED, CRect());
	m_selectableControl = new CMenuItemSelectable(    this, CControl::CCONTROL_NO_ID_REQUIRED, CRect(), NULL);

	// Store the menu item, this will setup our children
	this->setMenuItem(menuItem);
	//m_titleLabel->setTextFormat(CGraphics::e_defaultLabelText);
	this->onDrawDrawThePanelBounds(false);
}

//	===========================================================================
CMenuItemDisplay::~CMenuItemDisplay()
{
	EXPONENT_CLASS_DESTRUCTION(CMenuItemDisplay);
}

//	===========================================================================
void CMenuItemDisplay::setMenuItem(CMenuItem *item)
{
	if (item == NULL)
	{
		throw CException("Menu item is null", "CMenuItemDisplay::setMenuItem(CMenuItem *)");
	}

	// Store the item
	m_menuItem = item;

	if (m_menuItem.getPointer())
	{
		// Remove all previous controls
		this->clearControls();

		IImage *theImage = NULL;
		if (m_menuItem->getType() == CMenuItem::e_menuItemSelectable)
		{
			CSelectableMenuItem *myItem = (CSelectableMenuItem *)m_menuItem.getMutablePointer();
			theImage = (m_useIcons) ? ((m_mouseOver) ?  m_menuItem->getRolloverImage() : (myItem->isSelected()) ? myItem->getNormalImage() : myItem->getUnCheckedImage()) : NULL;
		}
		else
		{
			theImage = (m_useIcons) ? ((m_mouseOver) ? m_menuItem->getRolloverImage() : m_menuItem->getNormalImage()) : NULL;
		}			
		m_imageLabel->setPrimaryImage(theImage);

		CRect iconArea(0, 0, (theImage) ? theImage->getImageSize().getWidth() : m_area.getHeight(), m_area.getHeight());													// Area for icon if necessary
		CRect pointerArea(m_area.getWidth() - m_area.getHeight(), 0, m_area.getHeight(), m_area.getHeight());																// Area for the pointer if necessary
		CRect labelAreaIcon((theImage) ? theImage->getImageSize().getWidth() : m_area.getHeight(), 0, m_area.getWidth() - m_area.getHeight(), m_area.getHeight());			// Are when only icon showing
		CRect labelAreaPointerIcon(m_area.getHeight(), 0, m_area.getWidth() - m_area.getHeight(), m_area.getHeight());														// Are when icon and pointer showing
		CRect labelAreaPointer(0, 0, m_area.getWidth() - m_area.getHeight(), m_area.getHeight());																			// Area when only pointer is showing
		CRect labelArea(0, 0, m_area.getWidth(), m_area.getHeight());																										// When nothing is showing

		// Offset all areas
		iconArea.inset(2);
		pointerArea.inset(2);
		labelAreaIcon.inset(2);
		labelAreaPointerIcon.inset(2);
		labelAreaPointer.inset(2);
		labelArea.inset(2);

		// Setup the text colour
		CAlphaColour textColour = m_colours.m_disabledTextColour;
		if (m_menuItem->isEnabled())
		{
			if ((m_menuItem->overrideTextColour()))
			{
				textColour = m_menuItem->getTextColour();
			}
			else
			{
				textColour = m_colours.m_textColour;
			}
		}

		// Set the text
		m_titleLabel->setText(m_menuItem->getText());
		m_titleLabel->setTextColour(textColour);
		m_subMenuPointer->setPointerColours((m_menuItem->isEnabled()) ? m_colours.m_subMenuPointerColour : m_colours.m_disabledTextColour);
		m_selectableControl->setMenuItem(NULL);
		
		// Add new ones
		switch(m_menuItem->getType())
		{
			// These items have an image (possibly), a text label (definately)
			case CMenuItem::e_menuItemNormal:
			case CMenuItem::e_menuItemHtmlLink:
				if (m_useIcons)
				{
					// Create them
					m_imageLabel->setArea(iconArea);
					m_titleLabel->setArea(labelAreaIcon);

					// Add the items
					this->addControl(m_imageLabel.getMutablePointer());
					this->addControl(m_titleLabel.getMutablePointer());
				}
				else
				{
					// We still have to use the offset position, so that our menu items line up
					if (m_hasSelect)
					{
						m_titleLabel->setArea(labelAreaIcon);
					}
					else
					{
						m_titleLabel->setArea(labelArea);
					}

					// Add the title label
					this->addControl(m_titleLabel.getMutablePointer());

					// Set the colour of the text 
					m_titleLabel->setTextColour((m_menuItem->isEnabled()) ? m_colours.m_textColour : m_colours.m_disabledTextColour);
				}
			break;
			case CMenuItem::e_subMenuHolder:
				if (m_useIcons)
				{
					// Create them
					m_imageLabel->setArea(iconArea);
					m_titleLabel->setArea(labelAreaPointerIcon);
					m_subMenuPointer->setArea(pointerArea);

					// Add the items
					this->addControl(m_imageLabel.getMutablePointer());
					this->addControl(m_titleLabel.getMutablePointer());
					this->addControl(m_subMenuPointer.getMutablePointer());
				}
				else
				{
					// We still have to use the offset position, so that our menu items line up
					m_titleLabel->setArea(labelAreaPointer);
					m_subMenuPointer->setArea(pointerArea);

					// Add the title label
					this->addControl(m_titleLabel.getMutablePointer());
					this->addControl(m_subMenuPointer.getMutablePointer());
				}
			break;
			case CMenuItem::e_menuItemSeperator:

			break;
			case CMenuItem::e_menuItemSelectable:
				{
					// Store the menu item
					m_selectableControl->setMenuItem(m_menuItem.getMutablePointer());

					// Create them
					m_selectableControl->setArea(iconArea);
					m_titleLabel->setArea(labelAreaIcon);

					// Add the items
					this->addControl(m_selectableControl.getMutablePointer());
					this->addControl(m_titleLabel.getMutablePointer());
				}
			break;
			case CMenuItem::e_verticalSeperator:

			break;
		}
	}
}

//	===========================================================================
void CMenuItemDisplay::mouseIsOver(const bool mouseOver) 
{ 
	m_mouseOver = mouseOver; 

	if (!m_menuItem.pointerIsValid())
	{
		return;
	}

	IImage *theImage = NULL;
	if (m_menuItem->getType() == CMenuItem::e_menuItemSelectable)
	{
		CSelectableMenuItem *myItem = (CSelectableMenuItem *)m_menuItem.getMutablePointer();
		theImage = (m_useIcons) ? ((m_mouseOver) ?  m_menuItem->getRolloverImage() : (myItem->isSelected()) ? myItem->getNormalImage() : myItem->getUnCheckedImage()) : NULL;
	}
	else
	{
		theImage = (m_useIcons) ? ((m_mouseOver) ? m_menuItem->getRolloverImage() : m_menuItem->getNormalImage()) : NULL;
	}			
	m_imageLabel->setPrimaryImage(theImage);
}

//	===========================================================================
void CMenuItemDisplay::setFont(CFont *font)
{
	m_titleLabel->setFont(font);
}

//	===========================================================================
void CMenuItemDisplay::setColours(const SMenuColours &colours)
{
	m_colours.m_disabledTextColour   = colours.m_disabledTextColour;
	m_colours.m_frameColour		     = colours.m_frameColour;
	m_colours.m_selectedColour	     = colours.m_selectedColour;
	m_colours.m_seperatorColour	     = colours.m_seperatorColour;
	m_colours.m_subMenuPointerColour = colours.m_subMenuPointerColour;
	m_colours.m_textColour			 = colours.m_textColour;
			
	m_titleLabel->setTextColour((m_menuItem.pointerIsValid() && m_menuItem->isEnabled()) ? m_colours.m_textColour : m_colours.m_disabledTextColour);
	m_subMenuPointer->setPointerColours((m_menuItem.pointerIsValid() && m_menuItem->isEnabled()) ? m_colours.m_subMenuPointerColour : m_colours.m_disabledTextColour);
}

//	===========================================================================
void CMenuItemDisplay::drawControl(CGraphics &graphics)
{
	if (!m_menuItem.pointerIsValid())
	{
		// Now draw the parent panel
		CControlPanel::drawControl(graphics);
		return;
	}

	// Forced update of the colour
	CAlphaColour textColour = m_colours.m_textColour;
	if (m_menuItem->isEnabled())
	{
		if (m_menuItem->overrideTextColour())
		{
			textColour = m_menuItem->getTextColour();
		}
	}
	else
	{
		textColour = m_colours.m_disabledTextColour;
	}

	// Set the text
	m_titleLabel->setTextColour(textColour);

	switch(m_menuItem->getType())
	{
		// These items have an image (possibly), a text label (definately)
		case CMenuItem::e_menuItemNormal:
		case CMenuItem::e_menuItemHtmlLink:
		case CMenuItem::e_subMenuHolder:
		case CMenuItem::e_menuItemSelectable:
			CControlPanel::drawControl(graphics);
		break;
		case CMenuItem::e_menuItemSeperator:
			graphics.getMutablePen()->setColour(m_colours.m_seperatorColour);
			graphics.drawLine(CPoint(2, m_area.getHeight() / 2), CPoint(m_area.getWidth() - 2, m_area.getHeight() / 2));
		break;
		case CMenuItem::e_verticalSeperator:
			// Do nothing, taken care of by layout
		break;
	}
}

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CMenuItemDisplay::CMenuItemSubMenuPointer, CControl);

//	===========================================================================
CMenuItemDisplay::CMenuItemSubMenuPointer::CMenuItemSubMenuPointer(IControlRoot *root, const long uniqueId, const CRect &area)
										 : CControl(root, uniqueId, area)
										 , m_pointerColour(CAlphaColour::CALPHACOLOUR_BLACK)
{
	EXPONENT_CLASS_CONSTRUCTION(CMenuItemDisplay::CMenuItemSubMenuPointer);
}

//	===========================================================================
CMenuItemDisplay::CMenuItemSubMenuPointer::~CMenuItemSubMenuPointer()
{
	EXPONENT_CLASS_DESTRUCTION(CMenuItemDisplay::CMenuItemSubMenuPointer);
}

//	===========================================================================
void CMenuItemDisplay::CMenuItemSubMenuPointer::drawControl(CGraphics &graphics)
{
	// First check if we can allow the standard handler to draw the disabled control
	if (!this->drawEnabledControl(graphics))
	{
		return;
	}

	CRect area = m_normalisedArea;
	area.inset(4);

	CPolygon polygon(4);

	TCountedPointerArray<CPoint> *array = polygon.getMutablePointArray();
	array->addElement(new CPoint(area.getLeft(),  area.getTop()));
	array->addElement(new CPoint(area.getRight(), area.getCentralYPosition()));
	array->addElement(new CPoint(area.getLeft(),  area.getBottom()));
	array->addElement(new CPoint(area.getLeft(),  area.getTop()));

	graphics.getMutablePen()->setColour(m_pointerColour);
	graphics.getMutableBrush()->setColour(m_pointerColour);
	graphics.fillPolygon(polygon);
}

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CMenuItemDisplay::CMenuItemSelectable, CControl);

//	===========================================================================
CMenuItemDisplay::CMenuItemSelectable::CMenuItemSelectable(IControlRoot *root, const long uniqueId, const CRect &area, CMenuItem *item)
									 : CControl(root, uniqueId, area)
{
	EXPONENT_CLASS_CONSTRUCTION(CMenuItemDisplay::CMenuItemSelectable);
	m_menuItem = item;
}

//	===========================================================================
CMenuItemDisplay::CMenuItemSelectable::~CMenuItemSelectable()
{
	EXPONENT_CLASS_DESTRUCTION(CMenuItemDisplay::CMenuItemSelectable);
}

//	===========================================================================
void CMenuItemDisplay::CMenuItemSelectable::drawControl(CGraphics &graphics)
{
	// First check if we can allow the standard handler to draw the disabled control
	if (!this->drawEnabledControl(graphics))
	{
		return;
	}

	if (m_menuItem.pointerIsValid())
	{
		CSelectableMenuItem *item = (CSelectableMenuItem *)m_menuItem.getMutablePointer();
		if (item->isSelected())
		{
			// Set the btrush
			graphics.getMutableBrush()->setColour(CAlphaColour::CALPHACOLOUR_BLACK);//m_colours.m_subMenuPointerColour);

			const long heightOfTick = 7;
			const long widthOfTick  = 7;

			long tickTop  = m_area.getCentralYPosition() - 4;
			long tickLeft = m_area.getCentralXPosition() - 4;

			graphics.fillRectangle(CRect(tickLeft++, tickTop++, 1, 3));
			graphics.fillRectangle(CRect(tickLeft++, tickTop++, 1, 3));
			graphics.fillRectangle(CRect(tickLeft++, tickTop--, 1, 3));
			graphics.fillRectangle(CRect(tickLeft++, tickTop--, 1, 3));
			graphics.fillRectangle(CRect(tickLeft++, tickTop--, 1, 3));
			graphics.fillRectangle(CRect(tickLeft++, tickTop--, 1, 3));
			graphics.fillRectangle(CRect(tickLeft++, tickTop--, 1, 3));
		}
		else
		{
			
		}
	}
}